function [commLabels Q reducedMat] = getCommunitiesMM(obj, varargin)
%GETCOMMUNITIESMM Show the community structure of the AdjacencyMatrix obj
%using the modularity-maximization algorithm of Newman and Leicht, 2008
%   GETCOMMUNITIESMM(obj) will print out the list of communties, calling
%       the function modularity_dir of the brain connectivity toolbox.
%       This implements the algorithm presented in Newman and Leicht,
%       Community structure in directed networks.  This method will also
%       display a dendrogram representing the community splits that occured
%       in the Newman and Leicht algorithm.  Heights in the dendrogram
%       represent the improvement in modularity obtained by a split.
%
%   [commLabels Q reducedMat] = GETCOMMUNITIES(obj) will print out a list of
%       communities, and return a cell array of string arrays 'commLabels'.
%       Each element in 'commLabels' gives the members of that community. Q
%       is the modularity for this community structure.  reducedMat is an
%       AdjacencyMatrix object, giving a reduced matrix formed by grouping
%       all community members into meta-nodes.
%
%   Optional paremeters:
%       [ ... ] = GETCOMMUNITIES(obj,'quiet') will suppress text
%           output
%       [ ... ] = GETCOMMUNITIES(obj,'pTest') will perform a
%           statistical test of significance for the modularity and number
%           of modules, compared to a randomly rewired graph

quiet = false;
pTest = false;
for i = 1:numel(varargin)
    switch varargin{i}
        case 'quiet'
            quiet = true;
        case 'pTest'
            pTest = true;            
    end
end

[A labels] = getSquareMatrix(obj);

%   modularity_dir uses eig function; only works for full matrices
[Ci Q] = modularity_dir(full(A),~quiet);

if ~quiet, fprintf('Modularity of community structure: %.3f', Q); end

k = 1;
commLabels = {};
for i = 1:max(Ci)
    if ~quiet
        fprintf('\nCommunity %u\n',i);
        fprintf('%s, ',labels{Ci==i});
        fprintf('\n');
    end
    

    if ~isempty({labels{Ci==i}})
        
    % this caused some problems - ended up with empty matrices, which left
    % out some cells
%         try
%             matrices{k} = getSubMatrix(obj,{labels{Ci==i}});
%         catch ME
%             % do nothing; this is probably an empty matrix
%             k = k-1;
%         end

          commLabels{k} = {labels{Ci==i}};
    end
    k = k+1;
end

nIters = 1000;
nSwitches = 10;

% Now create reduced AdjacencyMatrix
groups = {};
for i = 1:numel(commLabels)
    metaNodeName = sprintf('Module %u',i);
    groups = [groups metaNodeName {commLabels{i}}];
end

reducedMat = combineCells(obj,groups,groups');

if pTest
    Qs = zeros(1,nIters);
    nModules = zeros(1,nIters);
    
    for i = 1:nIters
        A = randmio_dir_connected(getSquareMatrix(obj),nSwitches);
        [Ci Qs(i)] = modularity_dir(full(A));
        nModules(i) =  max(Ci);
    end
    
    meanQs = mean(Qs);
    sigQs = std(Qs);
    t = (Q-meanQs)/(sigQs*(sqrt(1+1/nIters)));
    pVal = sum(double(Qs>Q))/nIters;
    fprintf('\nAverage modularity of random graph: %.4f', meanQs);
    fprintf('\nSTD of modularity of random graphs: %.4f', sigQs);
    fprintf('\nt-score for this modularity: %f',t);
    fprintf('\nEstimated p value for this modularity: %f',pVal);
    fprintf('\np value calculated from student''s t distribution: %f\n', 1-tcdf(t,nIters-1));
    hist(Qs);
    title('Histogram of Q (maximum modularity) for random graphs');
    
    figure;
    hist(nModules);    
    title('Histogram of Number of Modules for random graphs');
    fprintf('\nAverage number of modules for random graphs: %.2f',mean(nModules));
    fprintf('\nSTD of number of modules for random graphs: %.4f\n',std(nModules));
end




end

